<?php
// Database connection
$host = 'localhost:3306';
$dbname = 'tawasoll_book';
$username = 'root';
$password = '';

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch(PDOException $e) {
    die("Connection failed: " . $e->getMessage());
}

// Get transaction ID
if (!isset($_GET['id'])) {
    header('Location: customer_bookkeeping.php');
    exit;
}

$transaction_id = $_GET['id'];

// Get transaction details
$stmt = $pdo->prepare("SELECT * FROM transactions WHERE id = :id");
$stmt->execute([':id' => $transaction_id]);
$transaction = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$transaction) {
    die('Transaction not found');
}

// Get all customers for dropdown
$customers_query = "SELECT customer_phone, name, business_type, preferred_currency FROM customer_context ORDER BY name";
$customers = $pdo->query($customers_query)->fetchAll(PDO::FETCH_ASSOC);

$success_message = '';
$error_message = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    try {
        $pdo->beginTransaction();
        
        // Get old transaction data
        $old_customer_phone = $transaction['customer_phone'];
        $old_transaction_type = $transaction['transaction_type'];
        $old_aed_amount = $transaction['aed_amount'];
        
        // Get new data
        $new_customer_phone = $_POST['customer_phone'];
        $new_transaction_type = $_POST['transaction_type'];
        $new_foreign_currency = $_POST['foreign_currency'] ?: null;
        $new_foreign_amount = $_POST['foreign_amount'] ?: null;
        $new_exchange_rate = $_POST['exchange_rate'] ?: null;
        $new_aed_amount = $_POST['aed_amount'];
        $new_description = $_POST['description'] ?: null;
        $new_notes = $_POST['notes'] ?: null;
        
        // Reverse old transaction effect on customer balance
        if ($old_transaction_type == 'CREDIT') {
            $reverse_old = "UPDATE customer_context 
                          SET current_balance = current_balance - :amount,
                              total_credits = total_credits - :amount,
                              transaction_count = transaction_count - 1
                          WHERE customer_phone = :phone";
        } else {
            $reverse_old = "UPDATE customer_context 
                          SET current_balance = current_balance + :amount,
                              total_debits = total_debits - :amount,
                              transaction_count = transaction_count - 1
                          WHERE customer_phone = :phone";
        }
        
        $reverse_stmt = $pdo->prepare($reverse_old);
        $reverse_stmt->execute([
            ':amount' => $old_aed_amount,
            ':phone' => $old_customer_phone
        ]);
        
        // Get new customer balance
        $customer_stmt = $pdo->prepare("SELECT current_balance FROM customer_context WHERE customer_phone = :phone");
        $customer_stmt->execute([':phone' => $new_customer_phone]);
        $customer = $customer_stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$customer) {
            throw new Exception('Customer not found');
        }
        
        $balance_before = $customer['current_balance'];
        
        // Calculate new balance after
        if ($new_transaction_type == 'CREDIT') {
            $balance_after = $balance_before + $new_aed_amount;
        } else {
            $balance_after = $balance_before - $new_aed_amount;
        }
        
        // Update transaction
        $update_query = "UPDATE transactions 
                        SET customer_phone = :customer_phone,
                            transaction_type = :transaction_type,
                            foreign_currency = :foreign_currency,
                            foreign_amount = :foreign_amount,
                            exchange_rate = :exchange_rate,
                            aed_amount = :aed_amount,
                            balance_before = :balance_before,
                            balance_after = :balance_after,
                            description = :description,
                            notes = :notes
                        WHERE id = :id";
        
        $update_stmt = $pdo->prepare($update_query);
        $update_stmt->execute([
            ':customer_phone' => $new_customer_phone,
            ':transaction_type' => $new_transaction_type,
            ':foreign_currency' => $new_foreign_currency,
            ':foreign_amount' => $new_foreign_amount,
            ':exchange_rate' => $new_exchange_rate,
            ':aed_amount' => $new_aed_amount,
            ':balance_before' => $balance_before,
            ':balance_after' => $balance_after,
            ':description' => $new_description,
            ':notes' => $new_notes,
            ':id' => $transaction_id
        ]);
        
        // Apply new transaction effect on customer balance
        if ($new_transaction_type == 'CREDIT') {
            $apply_new = "UPDATE customer_context 
                         SET current_balance = current_balance + :amount,
                             total_credits = total_credits + :amount,
                             transaction_count = transaction_count + 1
                         WHERE customer_phone = :phone";
        } else {
            $apply_new = "UPDATE customer_context 
                         SET current_balance = current_balance - :amount,
                             total_debits = total_debits + :amount,
                             transaction_count = transaction_count + 1
                         WHERE customer_phone = :phone";
        }
        
        $apply_stmt = $pdo->prepare($apply_new);
        $apply_stmt->execute([
            ':amount' => $new_aed_amount,
            ':phone' => $new_customer_phone
        ]);
        
        $pdo->commit();
        
        $success_message = 'Transaction updated successfully!';
        
        // Refresh transaction data
        $stmt = $pdo->prepare("SELECT * FROM transactions WHERE id = :id");
        $stmt->execute([':id' => $transaction_id]);
        $transaction = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Redirect after 2 seconds
        header("refresh:2;url=customer_bookkeeping.php?customer=" . urlencode($new_customer_phone));
        
    } catch(Exception $e) {
        $pdo->rollBack();
        $error_message = 'Error: ' . $e->getMessage();
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit Transaction - Tawasoll Book</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css" rel="stylesheet">
    <style>
        body {
            background-color: #f8f9fa;
        }
        .header-section {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 2rem 0;
            margin-bottom: 2rem;
        }
        .form-container {
            background: white;
            border-radius: 10px;
            padding: 2rem;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .required::after {
            content: " *";
            color: red;
        }
        .info-badge {
            background-color: #e3f2fd;
            padding: 1rem;
            border-radius: 5px;
            margin-bottom: 1.5rem;
        }
    </style>
</head>
<body>
    <div class="header-section">
        <div class="container">
            <h1 class="mb-0"><i class="bi bi-pencil"></i> Edit Transaction</h1>
            <p class="mb-0">Tawasoll Book - Update Transaction Details</p>
        </div>
    </div>

    <div class="container">
        <div class="row justify-content-center">
            <div class="col-lg-8">
                <?php if ($success_message): ?>
                <div class="alert alert-success alert-dismissible fade show">
                    <i class="bi bi-check-circle"></i> <?= $success_message ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>
                
                <?php if ($error_message): ?>
                <div class="alert alert-danger alert-dismissible fade show">
                    <i class="bi bi-exclamation-triangle"></i> <?= $error_message ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>

                <div class="info-badge">
                    <strong>Transaction ID:</strong> <?= $transaction['id'] ?> | 
                    <strong>Created:</strong> <?= date('Y-m-d H:i', strtotime($transaction['transaction_date'])) ?>
                </div>

                <div class="form-container">
                    <form method="POST" action="" id="transactionForm">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label required">Customer</label>
                                <select name="customer_phone" class="form-select" required id="customerSelect">
                                    <option value="">Select Customer</option>
                                    <?php foreach ($customers as $customer): ?>
                                        <option value="<?= htmlspecialchars($customer['customer_phone']) ?>"
                                                data-currency="<?= htmlspecialchars($customer['preferred_currency']) ?>"
                                                <?= $transaction['customer_phone'] == $customer['customer_phone'] ? 'selected' : '' ?>>
                                            <?= htmlspecialchars($customer['name'] ?: $customer['customer_phone']) ?> 
                                            (<?= htmlspecialchars($customer['business_type'] ?: 'N/A') ?>)
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label class="form-label required">Transaction Type</label>
                                <select name="transaction_type" class="form-select" required>
                                    <option value="">Select Type</option>
                                    <option value="CREDIT" <?= $transaction['transaction_type'] == 'CREDIT' ? 'selected' : '' ?>>
                                        CREDIT (Money In)
                                    </option>
                                    <option value="DEBIT" <?= $transaction['transaction_type'] == 'DEBIT' ? 'selected' : '' ?>>
                                        DEBIT (Money Out)
                                    </option>
                                </select>
                            </div>

                            <div class="col-md-4 mb-3">
                                <label class="form-label">Foreign Currency</label>
                                <input type="text" name="foreign_currency" class="form-control" 
                                       placeholder="USD, EUR, etc." id="foreignCurrency"
                                       value="<?= htmlspecialchars($transaction['foreign_currency']) ?>">
                            </div>

                            <div class="col-md-4 mb-3">
                                <label class="form-label">Foreign Amount</label>
                                <input type="number" step="0.01" name="foreign_amount" class="form-control" 
                                       placeholder="0.00" id="foreignAmount"
                                       value="<?= htmlspecialchars($transaction['foreign_amount']) ?>">
                            </div>

                            <div class="col-md-4 mb-3">
                                <label class="form-label">Exchange Rate</label>
                                <input type="number" step="0.0001" name="exchange_rate" class="form-control" 
                                       placeholder="0.0000" id="exchangeRate"
                                       value="<?= htmlspecialchars($transaction['exchange_rate']) ?>">
                            </div>

                            <div class="col-md-12 mb-3">
                                <label class="form-label required">AED Amount</label>
                                <div class="input-group">
                                    <input type="number" step="0.01" name="aed_amount" class="form-control" 
                                           placeholder="0.00" required id="aedAmount"
                                           value="<?= htmlspecialchars($transaction['aed_amount']) ?>">
                                    <span class="input-group-text">AED</span>
                                    <button class="btn btn-outline-secondary" type="button" id="calculateBtn">
                                        <i class="bi bi-calculator"></i> Calculate
                                    </button>
                                </div>
                                <small class="text-muted">Enter amount in AED or calculate from foreign currency</small>
                            </div>

                            <div class="col-md-12 mb-3">
                                <label class="form-label">Description</label>
                                <input type="text" name="description" class="form-control" 
                                       placeholder="Brief description of transaction"
                                       value="<?= htmlspecialchars($transaction['description']) ?>">
                            </div>

                            <div class="col-md-12 mb-3">
                                <label class="form-label">Notes</label>
                                <textarea name="notes" class="form-control" rows="3" 
                                          placeholder="Additional notes or comments"><?= htmlspecialchars($transaction['notes']) ?></textarea>
                            </div>
                        </div>

                        <div class="alert alert-warning">
                            <i class="bi bi-exclamation-triangle"></i> <strong>Warning:</strong> Editing this transaction will update customer balances and totals accordingly.
                        </div>

                        <div class="d-flex justify-content-between mt-4">
                            <a href="customer_bookkeeping.php?customer=<?= urlencode($transaction['customer_phone']) ?>" 
                               class="btn btn-secondary">
                                <i class="bi bi-arrow-left"></i> Back to List
                            </a>
                            <div>
                                <button type="reset" class="btn btn-outline-secondary">
                                    <i class="bi bi-x-circle"></i> Reset
                                </button>
                                <button type="submit" class="btn btn-primary">
                                    <i class="bi bi-save"></i> Update Transaction
                                </button>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Calculate AED amount from foreign currency
        document.getElementById('calculateBtn').addEventListener('click', function() {
            const foreignAmount = parseFloat(document.getElementById('foreignAmount').value) || 0;
            const exchangeRate = parseFloat(document.getElementById('exchangeRate').value) || 0;
            
            if (foreignAmount && exchangeRate) {
                const aedAmount = foreignAmount * exchangeRate;
                document.getElementById('aedAmount').value = aedAmount.toFixed(2);
            } else {
                alert('Please enter both foreign amount and exchange rate');
            }
        });

        // Auto-calculate when foreign amount or exchange rate changes
        document.getElementById('foreignAmount').addEventListener('input', autoCalculate);
        document.getElementById('exchangeRate').addEventListener('input', autoCalculate);

        function autoCalculate() {
            const foreignAmount = parseFloat(document.getElementById('foreignAmount').value) || 0;
            const exchangeRate = parseFloat(document.getElementById('exchangeRate').value) || 0;
            
            if (foreignAmount && exchangeRate) {
                const aedAmount = foreignAmount * exchangeRate;
                document.getElementById('aedAmount').value = aedAmount.toFixed(2);
            }
        }
    </script>
</body>
</html>
