# Customer Bookkeeping System - Tawasoll Book

A comprehensive PHP-based bookkeeping system for managing customer transactions with full CRUD operations, filtering, and export capabilities.

## Features

✅ **Customer Management**
- Dropdown list of all customers
- View customer details and current balance
- Auto-populate preferred currency

✅ **Transaction Management**
- Add new transactions (Credit/Debit)
- Edit existing transactions
- Delete transactions with balance recalculation
- Copy/duplicate transactions
- View detailed transaction information

✅ **Filtering & Search**
- Filter by customer
- Filter by date range
- Search by description, notes, currency
- Pagination (20 records per page)

✅ **Financial Tracking**
- Running balance calculation
- Summary cards (Total Credits, Total Debits, Net Balance, Transaction Count)
- Balance before/after for each transaction
- Multi-currency support with exchange rates

✅ **Export Functionality**
- Export to Excel (.xlsx)
- Export to PDF
- Formatted reports with summaries

✅ **User Interface**
- Modern Bootstrap 5 design
- Responsive layout (mobile-friendly)
- Color-coded transaction types (Credit = Green, Debit = Red)
- Interactive modals for viewing details
- Real-time calculations

## Files Included

1. **customer_bookkeeping.php** - Main transaction listing page
2. **add_transaction.php** - Add new transactions
3. **edit_transaction.php** - Edit existing transactions
4. **get_transaction.php** - AJAX endpoint for transaction details
5. **delete_transaction.php** - Delete transaction handler
6. **README.md** - This file

## Database Structure

### Tables Used

**customer_context**
- customer_phone (Primary Key)
- name
- business_type
- current_balance
- total_credits
- total_debits
- transaction_count
- preferred_currency
- notes
- created_at
- last_transaction_date

**transactions**
- id (Primary Key, Auto-increment)
- customer_phone (Foreign Key)
- transaction_date
- transaction_type (CREDIT/DEBIT)
- foreign_currency
- foreign_amount
- exchange_rate
- aed_amount
- balance_before
- balance_after
- description
- notes

## Installation

### Prerequisites
- PHP 7.4 or higher
- MySQL 5.7 or higher
- Web server (Apache/Nginx)
- Modern web browser

### Setup Steps

1. **Database Configuration**
   
   Update the database connection settings in all PHP files:
   ```php
   $host = 'localhost:3306';
   $dbname = 'tawasoll_book';
   $username = 'root';  // Change to your MySQL username
   $password = '';      // Change to your MySQL password
   ```

2. **Upload Files**
   
   Copy all PHP files to your web server directory:
   - XAMPP: `C:/xampp/htdocs/tawasoll_book/`
   - WAMP: `C:/wamp64/www/tawasoll_book/`
   - Linux: `/var/www/html/tawasoll_book/`

3. **Database Setup**
   
   Your database `tawasoll_book` should already exist with:
   - `customer_context` table
   - `transactions` table

4. **Access the Application**
   
   Open your browser and navigate to:
   ```
   http://localhost/tawasoll_book/customer_bookkeeping.php
   ```

## Usage Guide

### Viewing Transactions

1. Open `customer_bookkeeping.php`
2. Use filters to narrow down results:
   - Select customer from dropdown
   - Set date range
   - Enter search keywords
3. Click "Filter" to apply
4. Click "Reset" to clear all filters

### Adding Transactions

1. Click "Add Transaction" button
2. Fill in the form:
   - Select customer
   - Choose transaction type (CREDIT/DEBIT)
   - Enter foreign currency details (optional)
   - Enter AED amount or use calculator
   - Add description and notes
3. Click "Save Transaction"

### Editing Transactions

1. Click the edit (pencil) icon on any transaction
2. Modify the details
3. Click "Update Transaction"
4. Customer balances will be automatically recalculated

### Copying Transactions

1. Click the copy (clipboard) icon
2. System will pre-fill the add form with transaction details
3. Modify as needed
4. Save as new transaction

### Deleting Transactions

1. Click the delete (trash) icon
2. Confirm deletion
3. Transaction will be removed and balances recalculated

### Exporting Data

**Excel Export:**
- Click "Export to Excel" button
- File will download as `transactions_YYYY-MM-DD.xlsx`

**PDF Export:**
- Click "Export to PDF" button
- File will download as `transactions_YYYY-MM-DD.pdf`

## Features Explained

### Running Balance
- Each transaction shows balance before and after
- Automatically calculated based on transaction type
- Credits add to balance, debits subtract

### Multi-Currency Support
- Enter transactions in foreign currencies
- Specify exchange rate
- System calculates AED equivalent
- Auto-calculation when both amount and rate are entered

### Summary Cards
- **Total Credits**: Sum of all CREDIT transactions in filter
- **Total Debits**: Sum of all DEBIT transactions in filter
- **Net Balance**: Credits minus Debits
- **Transaction Count**: Total number of transactions

### Color Coding
- Green rows = CREDIT transactions (money in)
- Red rows = DEBIT transactions (money out)
- Easy visual identification

### Customer Info Card
When a customer is selected, displays:
- Customer name and phone
- Business type
- Preferred currency
- Current balance
- Total transaction count

## Security Notes

⚠️ **Important Security Considerations:**

1. **SQL Injection Protection**: All queries use prepared statements
2. **XSS Protection**: All output is escaped with `htmlspecialchars()`
3. **Transaction Integrity**: Database transactions ensure balance consistency
4. **Authentication**: Add user authentication for production use
5. **Access Control**: Implement role-based access control

## Customization

### Changing Records Per Page
Edit `customer_bookkeeping.php`:
```php
$per_page = 20;  // Change to desired number
```

### Modifying Date Format
Change date display format:
```php
date('Y-m-d H:i', strtotime($trans['transaction_date']))
```

### Adding New Transaction Types
Currently supports CREDIT and DEBIT. To add more:
1. Update database enum
2. Modify form dropdowns
3. Update balance calculation logic

### Styling Changes
- Edit inline `<style>` sections
- Colors, fonts, spacing can all be customized
- Bootstrap classes can be modified

## Troubleshooting

### Common Issues

**Problem: Connection failed**
- Check database credentials
- Ensure MySQL server is running
- Verify database name is correct

**Problem: Transactions not showing**
- Check if database has data
- Verify filter settings aren't too restrictive
- Check customer selection

**Problem: Balance calculation errors**
- Ensure transaction types are correct
- Check for manual database edits
- Recalculate all balances from scratch

**Problem: Export not working**
- Check browser console for errors
- Ensure JavaScript libraries are loading
- Allow pop-ups/downloads in browser

## Browser Compatibility

✅ Chrome/Edge (recommended)
✅ Firefox
✅ Safari
✅ Opera

## Performance

- Pagination limits database load
- Indexed database queries
- Optimized for 1000+ transactions
- AJAX for quick actions (view, delete)

## Future Enhancements

Potential additions:
- User authentication system
- Report generation (monthly, yearly)
- Email notifications
- Receipt generation
- Multi-company support
- API endpoints
- Mobile app version
- Barcode/QR code integration

## Support

For issues or questions:
1. Check this README
2. Review code comments
3. Check browser console for errors
4. Verify database structure

## License

This system is provided as-is for the Tawasoll Book project.

## Credits

Built with:
- PHP
- MySQL
- Bootstrap 5
- Bootstrap Icons
- SheetJS (Excel export)
- jsPDF (PDF export)

---

**Version:** 1.0  
**Last Updated:** January 2026  
**Database:** tawasoll_book
